/* SPDX-License-Identifier: GPL-2.0
 * aw882xx_init.h
 *
 * Copyright (c) 2024 AWINIC Technology CO., LTD
 *
 * Author: Nick Li <liweilei@awinic.com.cn>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 */

#ifndef __AW882XX_INIT_H__
#define __AW882XX_INIT_H__


#define AW_SOFT_RST_REG (0x00)
#define AW_SYSST_REG (0x01)
#define AW_SYSINTM_REG (0x03)
#define AW_SYSINT_REG  (0x02)
#define AW_SYSCTRL_REG (0x04)
#define AW_SYSCTRL2_REG	(0x05)
#define AW_I2SCTRL1_REG (0x06)
#define AW_DBGCTRL_REG (0x1F)
#define AW_VBAT_REG	(0x21)
#define AW_TEMP_REG (0x22)
#define AW_TESTIN_REG (0x38)
#define AW_TESTOUT_REG (0x39)
#define AW_NGCTRL3_REG		(0x5E)
#define AW_BSTCTRL2_REG		(0x61)
#define AW_LOCKH_REG (0x78)
#define AW_LOCKL_REG (0x7C)

#define AW_SYSINTM_DEFAULT (0xFFFF)
#define AW_SOFT_RST_VAL (0x55aa)

enum aw_chipid {
	PID_1852_ID = 0x1852,
	PID_2013_ID = 0x2013,
	PID_2032_ID = 0x2032,
	PID_2055_ID = 0x2055,
	PID_2055A_ID = 0x2055A,
	PID_2071_ID = 0x2071,
	PID_2113_ID = 0x2113,
	PID_2116_ID = 0x2116,
	PID_2308_ID = 0x2308,
	PID_2329_ID = 0x2329,
	PID_2330_ID = 0x2330,
	PID_2407_ID = 0x2407,
	PID_2418_ID = 0x2418,
};

#define AW_DEV_ATTR(dev_pid, dev_init_ptr) \
{	.pid = (dev_pid), \
	.dev_init = (dev_init_ptr)}

/********************************************
 * Volume Coefficient
 *******************************************/
#define AW_VOL_STEP		(64)
#define AW_MUTE_VOL		(1023)
#define AW_6_0P5_VOL_STEP_DB  (6 * 2)
#define AW_6_0P125_VOL_STEP_DB  (6 * 8)

/*monitor reg*/
#define AW_MONITOR_VBAT_RANGE		(6025)
#define AW_MONITOR_INT_10BIT		(1023)  // 1 << 10 -1
#define AW_MONITOR_TEMP_SIGN_MASK	(~(1<<9))
#define AW_MONITOR_TEMP_NEG_MASK	(0xFC00)


/* SYSST (0x01) detail */
/* UVLS bit 14 (SYSST 0x01) */
#define AW_UVLS_START_BIT		(14)
#define AW_UVLS_UVLO			(1)
#define AW_UVLS_UVLO_VALUE		\
	(AW_UVLS_UVLO << AW_UVLS_START_BIT)

/* BSTOCS bit 11 (SYSST 0x01) */
#define AW_BSTOCS_START_BIT	(11)
#define AW_BSTOCS_OVER_CURRENT	(1)
#define AW_BSTOCS_OVER_CURRENT_VALUE	\
	(AW_BSTOCS_OVER_CURRENT << AW_BSTOCS_START_BIT)

/* BSTS bit 9 (SYSST 0x01) */
#define AW_BSTS_START_BIT		(9)
#define AW_BSTS_FINISHED		(1)
#define AW_BSTS_FINISHED_VALUE	\
	(AW_BSTS_FINISHED << AW_BSTS_START_BIT)

/* SWS bit 8 (SYSST 0x01) */
#define AW_SWS_START_BIT		(8)
#define AW_SWS_SWITCHING		(1)
#define AW_SWS_SWITCHING_VALUE	\
	(AW_SWS_SWITCHING << AW_SWS_START_BIT)


/* NOCLKS bit 5 (SYSST 0x01) */
#define AW_NOCLKS_START_BIT	(5)
#define AW_NOCLKS_NO_CLOCK		(1)
#define AW_NOCLKS_NO_CLOCK_VALUE	\
	(AW_NOCLKS_NO_CLOCK << AW_NOCLKS_START_BIT)

/* CLKS bit 4 (SYSST 0x01) */
#define AW_CLKS_START_BIT		(4)
#define AW_CLKS_STABLE			(1)
#define AW_CLKS_STABLE_VALUE	\
	(AW_CLKS_STABLE << AW_CLKS_START_BIT)

/* OCDS bit 3 (SYSST 0x01) */
#define AW_OCDS_START_BIT		(3)
#define AW_OCDS_OC				(1)
#define AW_OCDS_OC_VALUE		\
	(AW_OCDS_OC << AW_OCDS_START_BIT)

/* OTHS bit 1 (SYSST 0x01) */
#define AW_OTHS_START_BIT		(1)
#define AW_OTHS_OT				(1)
#define AW_OTHS_OT_VALUE		\
	(AW_OTHS_OT << AW_OTHS_START_BIT)

/* PLLS bit 0 (SYSST 0x01) */
#define AW_PLLS_START_BIT		(0)
#define AW_PLLS_LOCKED			(1)
#define AW_PLLS_LOCKED_VALUE	\
	(AW_PLLS_LOCKED << AW_PLLS_START_BIT)


#define AW_SYSST_CHECK_MASK \
	(~(AW_UVLS_UVLO_VALUE | \
	AW_BSTOCS_OVER_CURRENT_VALUE | \
	AW_BSTS_FINISHED_VALUE | \
	AW_SWS_SWITCHING_VALUE | \
	AW_NOCLKS_NO_CLOCK_VALUE | \
	AW_CLKS_STABLE_VALUE | \
	AW_OCDS_OC_VALUE | \
	AW_OTHS_OT_VALUE | \
	AW_PLLS_LOCKED_VALUE))

#define AW_SWS_SYSST_CHECK \
	(AW_BSTS_FINISHED_VALUE | \
	AW_SWS_SWITCHING_VALUE | \
	AW_CLKS_STABLE_VALUE | \
	AW_PLLS_LOCKED_VALUE)

#define AW_NO_SWS_SYSST_CHECK \
		(AW_BSTS_FINISHED_VALUE | \
		AW_CLKS_STABLE_VALUE | \
		AW_PLLS_LOCKED_VALUE)

#define AW_IIS_CHECK \
	(AW_CLKS_STABLE_VALUE | \
	AW_PLLS_LOCKED_VALUE)

/* UVLI bit 14 (SYSINT 0x02) */
#define AW_UVLI_START_BIT		(14)
#define AW_UVLI_BITS_LEN		(1)
#define AW_UVLI_MASK			\
	(~(((1<<AW_UVLI_BITS_LEN)-1) << AW_UVLI_START_BIT))

/* BSTOCI bit 11 (SYSINT 0x02) */
#define AW_BSTOCI_START_BIT	(11)
#define AW_BSTOCI_BITS_LEN		(1)
#define AW_BSTOCI_MASK			\
	(~(((1<<AW_BSTOCI_BITS_LEN)-1) << AW_BSTOCI_START_BIT))

/* OCDI bit 3 (SYSINT 0x02) */
#define AW_OCDI_START_BIT		(3)
#define AW_OCDI_BITS_LEN		(1)
#define AW_OCDI_MASK			\
	(~(((1<<AW_OCDI_BITS_LEN)-1) << AW_OCDI_START_BIT))

/* OTHI bit 1 (SYSINT 0x02) */
#define AW_OTHI_START_BIT		(1)
#define AW_OTHI_BITS_LEN		(1)
#define AW_OTHI_MASK			\
	(~(((1<<AW_OTHI_BITS_LEN)-1) << AW_OTHI_START_BIT))

/* RCV_MODE bit 7 (SYSCTRL 0x04) */
#define AW_RCV_MODE_START_BIT	(7)
#define AW_RCV_MODE_BITS_LEN	(1)
#define AW_RCV_MODE_MASK		\
			(~(((1<<AW_RCV_MODE_BITS_LEN)-1) << AW_RCV_MODE_START_BIT))

#define AW_RCV_MODE_SPEAKER (0)
#define AW_RCV_MODE_SPEAKER_VALUE	\
			(AW_RCV_MODE_SPEAKER << AW_RCV_MODE_START_BIT)

#define AW_RCV_MODE_RECEIVER	(1)
#define AW_RCV_MODE_RECEIVER_VALUE	\
			(AW_RCV_MODE_RECEIVER << AW_RCV_MODE_START_BIT)


/* PWDN bit 0 (SYSCTRL 0x04) */
#define AW_PWDN_START_BIT		(0)
#define AW_PWDN_BITS_LEN		(1)
#define AW_PWDN_MASK			\
	(~(((1<<AW_PWDN_BITS_LEN)-1) << AW_PWDN_START_BIT))

#define AW_PWDN_DISABLE		(0)
#define AW_PWDN_DISABLE_VALUE	\
	(AW_PWDN_DISABLE << AW_PWDN_START_BIT)

#define AW_PWDN_ENABLE		(1)
#define AW_PWDN_ENABLE_VALUE	\
	(AW_PWDN_ENABLE << AW_PWDN_START_BIT)


/* AMPPD bit 1 (SYSCTRL 0x04) */
#define AW_AMPPD_START_BIT		(1)
#define AW_AMPPD_BITS_LEN		(1)
#define AW_AMPPD_MASK			\
		(~(((1<<AW_AMPPD_BITS_LEN)-1) << AW_AMPPD_START_BIT))

#define AW_AMPPD_ENABLE		(0)
#define AW_AMPPD_WORK_VALUE	\
		(AW_AMPPD_ENABLE << AW_AMPPD_START_BIT)

#define AW_AMPPD_DISABLE		(1)
#define AW_AMPPD_POWERDONE_VALUE	\
			(AW_AMPPD_DISABLE << AW_AMPPD_START_BIT)

/* HMUTE bit 8 (SYSCTRL 0x04) */
#define AW_HMUTE_START_BIT		(8)
#define AW_HMUTE_BITS_LEN		(1)
#define AW_HMUTE_MASK			\
			(~(((1<<AW_HMUTE_BITS_LEN)-1) << AW_HMUTE_START_BIT))

#define AW_HMUTE_ENABLE		(1)
#define AW_HMUTE_ENABLE_VALUE	\
			(AW_HMUTE_ENABLE << AW_HMUTE_START_BIT)

#define AW_HMUTE_DISABLE		(0)
#define AW_HMUTE_DISABLE_VALUE	\
				(AW_HMUTE_DISABLE << AW_HMUTE_START_BIT)


/* BOP_EN bit 12 (SYSCTRL 0x04) */
#define AW_BOP_EN_START_BIT	(12)
#define AW_BOP_EN_BITS_LEN		(1)
#define AW_BOP_EN_MASK			\
	(~(((1<<AW_BOP_EN_BITS_LEN)-1) << AW_BOP_EN_START_BIT))

#define AW_BOP_EN_DISABLE		(0)
#define AW_BOP_EN_DISABLE_VALUE	\
	(AW_BOP_EN_DISABLE << AW_BOP_EN_START_BIT)

#define AW_BOP_EN_ENABLE		(1)
#define AW_BOP_EN_ENABLE_VALUE	\
	(AW_BOP_EN_ENABLE << AW_BOP_EN_START_BIT)

/* ULS_HMUTE bit 15 (SYSCTRL 0x04) */
#define AW_ULS_HMUTE_START_BIT		(15)
#define AW_ULS_HMUTE_BITS_LEN		(1)
#define AW_ULS_HMUTE_MASK			\
				(~(((1<<AW_ULS_HMUTE_BITS_LEN)-1) << AW_ULS_HMUTE_START_BIT))

#define AW_ULS_HMUTE_ENABLE		(1)
#define AW_ULS_HMUTE_ENABLE_VALUE	\
				(AW_ULS_HMUTE_ENABLE << AW_ULS_HMUTE_START_BIT)

#define AW_ULS_HMUTE_DISABLE		(0)
#define AW_ULS_HMUTE_DISABLE_VALUE	\
					(AW_ULS_HMUTE_DISABLE << AW_ULS_HMUTE_START_BIT)

/* I2STXEN bit 13 (SYSCTRL 0x04) */
#define AW_I2STXEN_START_BIT	(13)
#define AW_I2STXEN_BITS_LEN	(1)
#define AW_I2STXEN_MASK		\
	(~(((1<<AW_I2STXEN_BITS_LEN)-1) << AW_I2STXEN_START_BIT))

#define AW_I2STXEN_DISABLE		(0)
#define AW_I2STXEN_DISABLE_VALUE	\
	(AW_I2STXEN_DISABLE << AW_I2STXEN_START_BIT)

#define AW_I2STXEN_ENABLE		(1)
#define AW_I2STXEN_ENABLE_VALUE	\
	(AW_I2STXEN_ENABLE << AW_I2STXEN_START_BIT)


/* VOL bit 9:0 (SYSCTRL2 0x05) */
#define FADE_DEFAULT_VOL_STEP (64)
#define AW_VOL_START_BIT	(0)
#define AW_VOL_BITS_LEN		(10)
#define AW_VOL_MASK			\
	(~(((1<<AW_VOL_BITS_LEN)-1) << AW_VOL_START_BIT))

/* CHSEL bit 11:10 (I2SCTRL1 0x06) */
#define AW_CHSEL_START_BIT		(10)
#define AW_CHSEL_BITS_LEN		(2)
#define AW_CHSEL_MASK			\
	(~(((1<<AW_CHSEL_BITS_LEN)-1) << AW_CHSEL_START_BIT))

#define AW_CHSEL_RESERVED		(0)
#define AW_CHSEL_RESERVED_VALUE	\
	(AW_CHSEL_RESERVED << AW_CHSEL_START_BIT)

#define AW_CHSEL_LEFT			(1)
#define AW_CHSEL_LEFT_VALUE	\
	(AW_CHSEL_LEFT << AW_CHSEL_START_BIT)

#define AW_CHSEL_RIGHT			(2)
#define AW_CHSEL_RIGHT_VALUE	\
	(AW_CHSEL_RIGHT << AW_CHSEL_START_BIT)

/* CCO_MUX bit 13 (DBGCTRL 0x1F) */
#define AW_CCO_MUX_START_BIT	(13)
#define AW_CCO_MUX_BITS_LEN	(1)
#define AW_CCO_MUX_MASK		\
	(~(((1<<AW_CCO_MUX_BITS_LEN)-1) << AW_CCO_MUX_START_BIT))

#define AW_CCO_MUX_DIVIDED		(0)
#define AW_CCO_MUX_DIVIDED_VALUE	\
	(AW_CCO_MUX_DIVIDED << AW_CCO_MUX_START_BIT)

#define AW_CCO_MUX_BYPASS		(1)
#define AW_CCO_MUX_BYPASS_VALUE	\
	(AW_CCO_MUX_BYPASS << AW_CCO_MUX_START_BIT)

#define AW_VOL_DEFAULT			(0)
#define AW_VOL_DEFAULT_VALUE	\
	(AW_VOL_DEFAULT << AW_VOL_START_BIT)

/* NOISE_GATE_EN bit 0 (NGCTRL3 0x5E) */
#define AW_NOISE_GATE_EN_START_BIT	(0)
#define AW_NOISE_GATE_EN_BITS_LEN	(1)
#define AW_NOISE_GATE_EN_MASK	\
	(~(((1<<AW_NOISE_GATE_EN_BITS_LEN)-1) << AW_NOISE_GATE_EN_START_BIT))

#define AW_NOISE_GATE_EN_DISABLE	(0)
#define AW_NOISE_GATE_EN_DISABLE_VALUE	\
		(AW_NOISE_GATE_EN_DISABLE << AW_NOISE_GATE_EN_START_BIT)

#define AW_NOISE_GATE_EN_ENABLE	(1)
#define AW_NOISE_GATE_EN_ENABLE_VALUE	\
		(AW_NOISE_GATE_EN_ENABLE << AW_NOISE_GATE_EN_START_BIT)

/* BST_IPEAK bit 15:12 (BSTCTRL2 0x61) */
#define AW_BST_IPEAK_START_BIT	(12)
#define AW_BST_IPEAK_BITS_LEN	(4)
#define AW_BST_IPEAK_MASK		\
	(~(((1<<AW_BST_IPEAK_BITS_LEN)-1) << AW_BST_IPEAK_START_BIT))


/* EF_LOCK bit 15 */
#define AW_EF_LOCK_STRAT_BIT (15)
#define AW_EF_LOCK_BITS_LEN (1)
#define AW_EF_LOCK_MASK \
	(~(((1<<AW_EF_LOCK_BITS_LEN)-1) << AW_EF_LOCK_STRAT_BIT))
#define AW_EF_LOCK_ENABLE (1)
#define AW_EF_LOCK_ENABLE_VALUE \
	(AW_EF_LOCK_ENABLE << AW_EF_LOCK_STRAT_BIT)

#ifndef __AW882XX_PID_1852_REG_H__
#define __AW882XX_PID_1852_REG_H__
#define AW_PID_1852_MONITOR_FILE	"aw882xx_pid_1852_monitor.bin"

#define AW_PID_1852_SYSCTRL2_REG (0x05)
#define AW_PID_1852_I2SCFG1_REG	(0x07)
#define AW_PID_1852_HAGCCFG4_REG (0x0c)
#define AW_PID_1852_VBAT_REG (0x12)
#define AW_PID_1852_TEMP_REG (0x13)
#define AW_PID_1852_VTMCTRL3_REG (0x56)
#define AW_PID_1852_PLLCTRL1_REG (0x66)
#define AW_PID_1852_TESTCTRL2_REG (0x71)
#define AW_PID_1852_EFRH_REG (0x78)
#define AW_PID_1852_EFRM1_REG (0x7A)
#define AW_PID_1852_REG_MAX				(0x7D)


/********************************************
 * vcalb
 *******************************************/
#define AW_PID_1852_VCAL_FACTOR (1 << 13)
#define AW_PID_1852_CABL_BASE_VALUE (1000)
#define AW_PID_1852_ICABLK_FACTOR (1)

#define AW_PID_1852_EF_VSN_GESLP_MASK		(~0x03ff)
#define AW_PID_1852_EF_VSN_GESLP_SIGN_MASK		(~0x0200)
#define AW_PID_1852_EF_VSN_GESLP_NEG		(~0xfc00)

#define AW_PID_1852_EF_ISN_GESLP_MASK		(~0x03ff)
#define AW_PID_1852_EF_ISN_GESLP_SIGN_MASK		(~0x0200)
#define AW_PID_1852_EF_ISN_GESLP_NEG		(~0xfc00)

#define AW_PID_1852_CABL_BASE_VALUE			(1000)
#define AW_PID_1852_ICABLK_FACTOR			(1)
#define AW_PID_1852_VCABLK_FACTOR			(1)


/* HMUTE bit 4 (SYSCTRL2 0x05) */
#define AW_PID_1852_HMUTE_START_BIT		(4)
#define AW_PID_1852_HMUTE_BITS_LEN		(1)
#define AW_PID_1852_HMUTE_MASK			\
	(~(((1<<AW_PID_1852_HMUTE_BITS_LEN)-1) << AW_PID_1852_HMUTE_START_BIT))

#define AW_PID_1852_HMUTE_DISABLE		(0)
#define AW_PID_1852_HMUTE_DISABLE_VALUE	\
	(AW_PID_1852_HMUTE_DISABLE << AW_PID_1852_HMUTE_START_BIT)

#define AW_PID_1852_HMUTE_ENABLE		(1)
#define AW_PID_1852_HMUTE_ENABLE_VALUE	\
	(AW_PID_1852_HMUTE_ENABLE << AW_PID_1852_HMUTE_START_BIT)


/* I2STXEN bit 0 (I2SCFG1 0x07) */
#define AW_PID_1852_I2STXEN_START_BIT	(0)
#define AW_PID_1852_I2STXEN_BITS_LEN	(1)
#define AW_PID_1852_I2STXEN_MASK		\
	(~(((1<<AW_PID_1852_I2STXEN_BITS_LEN)-1) << AW_PID_1852_I2STXEN_START_BIT))

#define AW_PID_1852_I2STXEN_DISABLE		(0)
#define AW_PID_1852_I2STXEN_DISABLE_VALUE	\
	(AW_PID_1852_I2STXEN_DISABLE << AW_PID_1852_I2STXEN_START_BIT)

#define AW_PID_1852_I2STXEN_ENABLE		(1)
#define AW_PID_1852_I2STXEN_ENABLE_VALUE	\
	(AW_PID_1852_I2STXEN_ENABLE << AW_PID_1852_I2STXEN_START_BIT)


/* I2S CCO_UMX */
#define AW_PID_1852_I2S_CCO_MUX_START_BIT		(14)
#define AW_PID_1852_I2S_CCO_MUX_BITS_LEN		(1)
#define AW_PID_1852_I2S_CCO_MUX_MASK			\
	(~(((1<<AW_PID_1852_I2S_CCO_MUX_BITS_LEN)-1)<<AW_PID_1852_I2S_CCO_MUX_START_BIT))

#define AW_PID_1852_I2S_CCO_MUX_8_16_32KHZ_BIT_VALUE		(0)
#define AW_PID_1852_I2S_CCO_MUX_8_16_32KHZ_VALUE			\
	(AW_PID_1852_I2S_CCO_MUX_8_16_32KHZ_BIT_VALUE<<AW_PID_1852_I2S_CCO_MUX_START_BIT)


#define AW_PID_1852_I2S_CCO_MUX_EXC_8_16_32KHZ_BIT_VALUE	(1)
#define AW_PID_1852_I2S_CCO_MUX_EXC_8_16_32KHZ_VALUE			\
	(AW_PID_1852_I2S_CCO_MUX_EXC_8_16_32KHZ_BIT_VALUE<<AW_PID_1852_I2S_CCO_MUX_START_BIT)


/* BST_IPEAK bit 3:0 (SYSCTRL2 0x05) */
#define AW_PID_1852_BST_IPEAK_START_BIT	(0)
#define AW_PID_1852_BST_IPEAK_BITS_LEN	(4)
#define AW_PID_1852_BST_IPEAK_MASK		\
	(~(((1<<AW_PID_1852_BST_IPEAK_BITS_LEN)-1) << AW_PID_1852_BST_IPEAK_START_BIT))


/********************************************
 * volume
 *******************************************/
/* VOL bit 15:8 (HAGCCFG4 0x0C) */
#define AW_PID_1852_MUTE_VOL  (90 * 2)
#define AW_PID_1852_VOL_STEP_DB   (6 * 2)

#define AW_PID_1852_VOL_START_BIT		(8)
#define AW_PID_1852_VOL_BITS_LEN		(8)
#define AW_PID_1852_VOL_MASK			\
	(~(((1<<AW_PID_1852_VOL_BITS_LEN)-1) << AW_PID_1852_VOL_START_BIT))

#define AW_PID_1852_VOL_DEFAULT_VALUE	(0)
#define AW_PID_1852_VOL_DEFAULT			\
	(AW_PID_1852_VOL_DEFAULT_VALUE << AW_PID_1852_VOL_START_BIT)


/********************************************
 * Dither
 *******************************************/
#define AW_PID_1852_DITHER_START_BIT	(7)
#define AW_PID_1852_DITHER_BITS_LEN	(1)
#define AW_PID_1852_DITHER_MASK	\
	(~(((1<<AW_PID_1852_DITHER_BITS_LEN)-1) << AW_PID_1852_DITHER_START_BIT))

#define AW_PID_1852_DITHER_DISABLE	(0)
#define AW_PID_1852_DITHER_DISABLE_VALUE	\
	(AW_PID_1852_DITHER_DISABLE << AW_PID_1852_DITHER_START_BIT)

#define AW_PID_1852_DITHER_ENABLE	(1)
#define AW_PID_1852_DITHER_ENABLE_VALUE	\
	(AW_PID_1852_DITHER_ENABLE << AW_PID_1852_DITHER_START_BIT)

#endif /*__AW882XX_PID_1852_REG_H__*/



#ifndef __AW882XX_PID_2013_REG_H__
#define __AW882XX_PID_2013_REG_H__

#define AW_PID_2013_REG_MAX  (0x7D)
#define	AW_PID_2013_MONITOR_FILE "aw882xx_pid_2013_monitor.bin"

#define AW_PID_2013_SYSCTRL_REG (0x04)
#define AW_PID_2013_I2SCTRL1_REG  (0x06)
#define AW_PID_2013_DBGCTRL_REG (0x12)
#define AW_PID_2013_VSNTM1_REG (0x50)
#define AW_PID_2013_PLLCTRL3_REG (0x55)
#define AW_PID_2013_BSTCTRL2_REG (0x61)
#define AW_PID_2013_EFRH_REG (0x78)
#define AW_PID_2013_EFRM2_REG (0x79)
#define AW_PID_2013_EFRM1_REG (0x7A)


/********************************************
 * Vcalb
 *******************************************/
#define AW_PID_2013_EF_ISN_GESLP2_SIGN_MASK		(~0x0200)
#define AW_PID_2013_EF_ISN_GESLP2_NEG		(~0xFC00)

#define AW_PID_2013_EF_ISN_GESLP_SIGN_MASK		(~0x0200)
#define AW_PID_2013_EF_ISN_GESLP_NEG		(~0xFC00)

#define AW_PID_2013_CABL_BASE_VALUE			(1000)
#define AW_PID_2013_ICABLK_FACTOR			(1)
#define AW_PID_2013_VCABLK_FACTOR			(1)

#define AW_PID_2013_VCAL_FACTOR			(1<<13)

/* I2STXEN bit 14 (I2SCTRL1 0x06) */
#define AW_PID_2013_I2STXEN_START_BIT	(14)
#define AW_PID_2013_I2STXEN_BITS_LEN	(1)
#define AW_PID_2013_I2STXEN_MASK	\
		(~(((1<<AW_PID_2013_I2STXEN_BITS_LEN)-1) << AW_PID_2013_I2STXEN_START_BIT))

#define AW_PID_2013_I2STXEN_DISABLE	(0)
#define AW_PID_2013_I2STXEN_DISABLE_VALUE	\
		(AW_PID_2013_I2STXEN_DISABLE << AW_PID_2013_I2STXEN_START_BIT)

#define AW_PID_2013_I2STXEN_ENABLE	(1)
#define AW_PID_2013_I2STXEN_ENABLE_VALUE	\
		(AW_PID_2013_I2STXEN_ENABLE << AW_PID_2013_I2STXEN_START_BIT)

/* EF_ISN_GESLP bit 9:0 (EFRH 0x78) */
#define AW_PID_2013_EF_ISN_GESLP_START_BIT	(0)
#define AW_PID_2013_EF_ISN_GESLP_BITS_LEN	(10)
#define AW_PID_2013_EF_ISN_GESLP_MASK	\
		(~(((1<<AW_PID_2013_EF_ISN_GESLP_BITS_LEN)-1) << AW_PID_2013_EF_ISN_GESLP_START_BIT))

/* EF_VSN_GESLP bit 9:0 (EFRM2 0x79) */
#define AW_PID_2013_EF_VSN_GESLP_START_BIT	(0)
#define AW_PID_2013_EF_VSN_GESLP_BITS_LEN	(10)
#define AW_PID_2013_EF_VSN_GESLP_MASK	\
		(~(((1<<AW_PID_2013_EF_VSN_GESLP_BITS_LEN)-1) << AW_PID_2013_EF_VSN_GESLP_START_BIT))

/* EN_TRAN bit 7 (SYSCTRL 0x04) */
#define AW_PID_2013_EN_TRAN_START_BIT	(7)
#define AW_PID_2013_EN_TRAN_BITS_LEN	(1)
#define AW_PID_2013_EN_TRAN_MASK	\
		(~(((1<<AW_PID_2013_EN_TRAN_BITS_LEN)-1) << AW_PID_2013_EN_TRAN_START_BIT))

#define AW_PID_2013_EN_TRAN_SPK	(0)
#define AW_PID_2013_EN_TRAN_SPK_VALUE	\
		(AW_PID_2013_EN_TRAN_SPK << AW_PID_2013_EN_TRAN_START_BIT)

/* BST_MODE bit 5:4 (BSTCTRL2 0x61) */
#define AW_PID_2013_BST_MODE_START_BIT	(4)
#define AW_PID_2013_BST_MODE_BITS_LEN	(2)
#define AW_PID_2013_BST_MODE_MASK	\
		(~(((1<<AW_PID_2013_BST_MODE_BITS_LEN)-1) << AW_PID_2013_BST_MODE_START_BIT))

#define AW_PID_2013_BST_MODE_TRANSPARENT	(0)
#define AW_PID_2013_BST_MODE_TRANSPARENT_VALUE	\
		(AW_PID_2013_BST_MODE_TRANSPARENT << AW_PID_2013_BST_MODE_START_BIT)

#define AW_PID_2013_BST_MODE_FORCE_BOOST	(1)
#define AW_PID_2013_BST_MODE_FORCE_BOOST_VALUE	\
		(AW_PID_2013_BST_MODE_FORCE_BOOST << AW_PID_2013_BST_MODE_START_BIT)

/* CCO_MUX bit 2 (PLLCTRL3 0x55) */
#define AW_PID_2013_CCO_MUX_START_BIT	(2)
#define AW_PID_2013_CCO_MUX_BITS_LEN	(1)
#define AW_PID_2013_CCO_MUX_MASK		\
		(~(((1<<AW_PID_2013_CCO_MUX_BITS_LEN)-1) << AW_PID_2013_CCO_MUX_START_BIT))

#define AW_PID_2013_CCO_MUX_DIVIDED		(0)
#define AW_PID_2013_CCO_MUX_DIVIDED_VALUE	\
		(AW_PID_2013_CCO_MUX_DIVIDED << AW_PID_2013_CCO_MUX_START_BIT)

#define AW_PID_2013_CCO_MUX_BYPASS		(1)
#define AW_PID_2013_CCO_MUX_BYPASS_VALUE	\
		(AW_PID_2013_CCO_MUX_BYPASS << AW_PID_2013_CCO_MUX_START_BIT)

#define AW_PID_2013_MUTE_VOL (90 * 8)

/********************************************
 * Dither
 *******************************************/
#define AW_PID_2013_DITHER_START_BIT	(14)
#define AW_PID_2013_DITHER_BITS_LEN	(1)
#define AW_PID_2013_DITHER_MASK	\
		(~(((1<<AW_PID_2013_DITHER_BITS_LEN)-1) << AW_PID_2013_DITHER_START_BIT))

#define AW_PID_2013_DITHER_DISABLE	(0)
#define AW_PID_2013_DITHER_DISABLE_VALUE	\
		(AW_PID_2013_DITHER_DISABLE << AW_PID_2013_DITHER_START_BIT)

#define AW_PID_2013_DITHER_ENABLE	(1)
#define AW_PID_2013_DITHER_ENABLE_VALUE	\
		(AW_PID_2013_DITHER_ENABLE << AW_PID_2013_DITHER_START_BIT)

#define AW_PID_2013_EFVER_CHECK		(0x0007)
#define AW_PID_2013_EFVERH_START_BIT	(13)
#define AW_PID_2013_EFVERL_START_BIT	(10)
#define AW_PID_2013_EFVERH_MASK		(~(7 << 13))
#define AW_PID_2013_EFVERL_MASK		(~(7 << 10))

#endif /*__AW882XX_PID_2013_REG_H__*/



#ifndef __AW882XX_PID_2032_REG_H__
#define __AW882XX_PID_2032_REG_H__

#define AW_PID_2032_REG_MAX (0x7E)
#define	AW_PID_2032_MONITOR_FILE "aw882xx_pid_2032_monitor.bin"

#define AW_PID_2032_SYSCTRL2_REG		(0x05)
#define AW_PID_2032_I2SCFG1_REG			(0x07)
#define AW_PID_2032_VBAT_REG			(0x12)
#define AW_PID_2032_TEMP_REG			(0x13)
#define AW_PID_2032_DBGCTRL_REG			(0x20)
#define AW_PID_2032_VTMCTRL3_REG		(0x56)
#define AW_PID_2032_PLLCTRL1_REG        (0x66)

#define AW_PID_2032_EFRH_REG			(0x78)

#define AW_PID_2032_EFRM1_REG			(0x7A)


/********************************************
 * Vcalb
 *******************************************/
#define AW_PID_2032_EF_VSN_GESLP_SIGN_MASK		(~0x0200)
#define AW_PID_2032_EF_VSN_GESLP_NEG		(~0xfc00)

#define AW_PID_2032_EF_ISN_GESLP_SIGN_MASK		(~0x0200)
#define AW_PID_2032_EF_ISN_GESLP_NEG		(~0xfc00)

#define AW_PID_2032_CABL_BASE_VALUE			(1000)
#define AW_PID_2032_ICABLK_FACTOR			(1)
#define AW_PID_2032_VCABLK_FACTOR			(1)

#define AW_PID_2032_VCAL_FACTOR			(1<<13)


/* HMUTE bit 4 (SYSCTRL2 0x05) */
#define AW_PID_2032_HMUTE_START_BIT		(4)
#define AW_PID_2032_HMUTE_BITS_LEN		(1)
#define AW_PID_2032_HMUTE_MASK			\
		(~(((1<<AW_PID_2032_HMUTE_BITS_LEN)-1) << AW_PID_2032_HMUTE_START_BIT))

#define AW_PID_2032_HMUTE_DISABLE		(0)
#define AW_PID_2032_HMUTE_DISABLE_VALUE	\
		(AW_PID_2032_HMUTE_DISABLE << AW_PID_2032_HMUTE_START_BIT)

#define AW_PID_2032_HMUTE_ENABLE		(1)
#define AW_PID_2032_HMUTE_ENABLE_VALUE	\
		(AW_PID_2032_HMUTE_ENABLE << AW_PID_2032_HMUTE_START_BIT)

/* I2STXEN bit 0 (I2SCFG1 0x07) */
#define AW_PID_2032_I2STXEN_START_BIT	(0)
#define AW_PID_2032_I2STXEN_BITS_LEN	(1)
#define AW_PID_2032_I2STXEN_MASK		\
		(~(((1<<AW_PID_2032_I2STXEN_BITS_LEN)-1) << AW_PID_2032_I2STXEN_START_BIT))

#define AW_PID_2032_I2STXEN_DISABLE		(0)
#define AW_PID_2032_I2STXEN_DISABLE_VALUE	\
		(AW_PID_2032_I2STXEN_DISABLE << AW_PID_2032_I2STXEN_START_BIT)

#define AW_PID_2032_I2STXEN_ENABLE		(1)
#define AW_PID_2032_I2STXEN_ENABLE_VALUE	\
		(AW_PID_2032_I2STXEN_ENABLE << AW_PID_2032_I2STXEN_START_BIT)

	/* EF_ISN_GESLP bit 9:0 (EFRM1 0x7A) */
#define AW_PID_2032_EF_ISN_GESLP_START_BIT	(0)
#define AW_PID_2032_EF_ISN_GESLP_BITS_LEN	(10)
#define AW_PID_2032_EF_ISN_GESLP_MASK	\
		(~(((1<<AW_PID_2032_EF_ISN_GESLP_BITS_LEN)-1) << AW_PID_2032_EF_ISN_GESLP_START_BIT))

/* EF_VSN_GESLP bit 9:0 (EFRH 0x78) */
#define AW_PID_2032_EF_VSN_GESLP_START_BIT	(0)
#define AW_PID_2032_EF_VSN_GESLP_BITS_LEN	(10)
#define AW_PID_2032_EF_VSN_GESLP_MASK		\
		(~(((1<<AW_PID_2032_EF_VSN_GESLP_BITS_LEN)-1) << AW_PID_2032_EF_VSN_GESLP_START_BIT))

/* CCO_MUX bit 14 (PLLCTRL1 0x66) */
#define AW_PID_2032_CCO_MUX_START_BIT	(14)
#define AW_PID_2032_CCO_MUX_BITS_LEN	(1)
#define AW_PID_2032_CCO_MUX_MASK		\
		(~(((1<<AW_PID_2032_CCO_MUX_BITS_LEN)-1) << AW_PID_2032_CCO_MUX_START_BIT))

#define AW_PID_2032_CCO_MUX_DIVIDED		(0)
#define AW_PID_2032_CCO_MUX_DIVIDED_VALUE	\
		(AW_PID_2032_CCO_MUX_DIVIDED << AW_PID_2032_CCO_MUX_START_BIT)

#define AW_PID_2032_CCO_MUX_BYPASS		(1)
#define AW_PID_2032_CCO_MUX_BYPASS_VALUE	\
		(AW_PID_2032_CCO_MUX_BYPASS << AW_PID_2032_CCO_MUX_START_BIT)

/* BST_IPEAK bit 3:0 (SYSCTRL2 0x05) */
#define AW_PID_2032_BST_IPEAK_START_BIT	(0)
#define AW_PID_2032_BST_IPEAK_BITS_LEN	(4)
#define AW_PID_2032_BST_IPEAK_MASK		\
		(~(((1<<AW_PID_2032_BST_IPEAK_BITS_LEN)-1) << AW_PID_2032_BST_IPEAK_START_BIT))

/* VOL bit 15:6 (SYSCTRL2 0x05) */
#define AW_PID_2032_VOL_START_BIT		(6)
#define AW_PID_2032_VOL_BITS_LEN		(10)
#define AW_PID_2032_VOL_MASK			\
		(~(((1<<AW_PID_2032_VOL_BITS_LEN)-1) << AW_PID_2032_VOL_START_BIT))

#define AW_PID_2032_MUTE_VOL	(90 * 8)

/********************************************
 * Dither
 *******************************************/
#define AW_PID_2032_DITHER_START_BIT	(13)
#define AW_PID_2032_DITHER_BITS_LEN	(1)
#define AW_PID_2032_DITHER_MASK	\
		(~(((1<<AW_PID_2032_DITHER_BITS_LEN)-1) << AW_PID_2032_DITHER_START_BIT))

#define AW_PID_2032_DITHER_DISABLE	(0)
#define AW_PID_2032_DITHER_DISABLE_VALUE	\
		(AW_PID_2032_DITHER_DISABLE << AW_PID_2032_DITHER_START_BIT)

#define AW_PID_2032_DITHER_ENABLE	(1)
#define AW_PID_2032_DITHER_ENABLE_VALUE	\
		(AW_PID_2032_DITHER_ENABLE << AW_PID_2032_DITHER_START_BIT)
#endif /*__AW882XX_PID_2032_REG_H__*/



#ifndef __AW882XX_PID_2055_REG_H__
#define __AW882XX_PID_2055_REG_H__
#define AW_PID_2055_REG_MAX  (0x7D)
#define AW_PID_2055_MONITOR_FILE	"aw882xx_pid_2055_monitor.bin"
#define AW_PID_2055A_MONITOR_FILE	"aw882xx_pid_2055a_monitor.bin"

#define AW_PID_2055A_VERSION_VALUE	(0x6708)
#define AW_PID_2055_VERSION_VALUE	(0x02A0)

#define AW_PID_2055_I2SCTRL1_REG        (0x06)
#define AW_PID_2055_DBGCTRL_REG			(0x17)
#define AW_PID_2055_SADCCTRL3_REG		(0x3C)
#define AW_PID_2055_PLLCTRL3_REG		(0x52)
#define AW_PID_2055_EFRH_REG			(0x78)
#define AW_PID_2055_EFRL_REG			(0x79)

#define AW_PID_2055_MUTE_VOL (90 * 8)
#define AW_PID_2055_INIT_CHECK_REG		(0x73)
#define AW_PID_2055_INIT_CHECK_VALUE		(0x1D40)

/* I2STXEN bit 14 (I2SCTRL1 0x06) */
#define AW_PID_2055_I2STXEN_START_BIT	(14)
#define AW_PID_2055_I2STXEN_BITS_LEN	(1)
#define AW_PID_2055_I2STXEN_MASK		\
		(~(((1<<AW_PID_2055_I2STXEN_BITS_LEN)-1) << AW_PID_2055_I2STXEN_START_BIT))

#define AW_PID_2055_I2STXEN_DISABLE		(0)
#define AW_PID_2055_I2STXEN_DISABLE_VALUE	\
		(AW_PID_2055_I2STXEN_DISABLE << AW_PID_2055_I2STXEN_START_BIT)

#define AW_PID_2055_I2STXEN_ENABLE		(1)
#define AW_PID_2055_I2STXEN_ENABLE_VALUE	\
		(AW_PID_2055_I2STXEN_ENABLE << AW_PID_2055_I2STXEN_START_BIT)

/* CCO_MUX bit 2 (PLLCTRL3 0x52) */
#define AW_PID_2055_CCO_MUX_START_BIT	(2)
#define AW_PID_2055_CCO_MUX_BITS_LEN	(1)
#define AW_PID_2055_CCO_MUX_MASK		\
		(~(((1<<AW_PID_2055_CCO_MUX_BITS_LEN)-1) << AW_PID_2055_CCO_MUX_START_BIT))

#define AW_PID_2055_CCO_MUX_DIVIDED		(0)
#define AW_PID_2055_CCO_MUX_DIVIDED_VALUE	\
		(AW_PID_2055_CCO_MUX_DIVIDED << AW_PID_2055_CCO_MUX_START_BIT)

#define AW_PID_2055_CCO_MUX_BYPASS		(1)
#define AW_PID_2055_CCO_MUX_BYPASS_VALUE	\
		(AW_PID_2055_CCO_MUX_BYPASS << AW_PID_2055_CCO_MUX_START_BIT)

/* BOP_EN bit 4 (SADCCTRL3 0x3C) */
#define AW_PID_2055_BOP_EN_START_BIT	(4)
#define AW_PID_2055_BOP_EN_BITS_LEN		(1)
#define AW_PID_2055_BOP_EN_MASK			\
		(~(((1<<AW_PID_2055_BOP_EN_BITS_LEN)-1) << AW_PID_2055_BOP_EN_START_BIT))

#define AW_PID_2055_BOP_EN_DISABLE		(0)
#define AW_PID_2055_BOP_EN_DISABLE_VALUE	\
		(AW_PID_2055_BOP_EN_DISABLE << AW_PID_2055_BOP_EN_START_BIT)

#define AW_PID_2055_BOP_EN_ENABLE		(1)
#define AW_PID_2055_BOP_EN_ENABLE_VALUE	\
		(AW_PID_2055_BOP_EN_ENABLE << AW_PID_2055_BOP_EN_START_BIT)

/********************************************
 * Dither
 *******************************************/
#define AW_PID_2055_DITHER_START_BIT	(14)
#define AW_PID_2055_DITHER_BITS_LEN	(1)
#define AW_PID_2055_DITHER_MASK	\
		(~(((1<<AW_PID_2055_DITHER_BITS_LEN)-1) << AW_PID_2055_DITHER_START_BIT))

#define AW_PID_2055_DITHER_DISABLE	(0)
#define AW_PID_2055_DITHER_DISABLE_VALUE	\
		(AW_PID_2055_DITHER_DISABLE << AW_PID_2055_DITHER_START_BIT)

#define AW_PID_2055_DITHER_ENABLE	(1)
#define AW_PID_2055_DITHER_ENABLE_VALUE	\
		(AW_PID_2055_DITHER_ENABLE << AW_PID_2055_DITHER_START_BIT)
#endif /*__AW882XX_PID_2055_REG_H__*/


#ifndef __AW882XX_PID_2071_REG_H__
#define __AW882XX_PID_2071_REG_H__

#define  AW_PID_2071_REG_MAX (0x7E)
#define AW_PID_2071_MONITOR_FILE	"aw882xx_pid_2071_monitor.bin"

#define AW_PID_2071_SYSCTRL2_REG	(0x05)
#define AW_PID_2071_I2SCFG1_REG		(0x07)
#define AW_PID_2071_VBAT_REG		(0x12)
#define AW_PID_2071_TEMP_REG		(0x13)
#define AW_PID_2071_DBGCTRL_REG		(0x20)
#define AW_PID_2071_VTMCTRL3_REG	(0x56)
#define AW_PID_2071_PLLCTRL1_REG	(0x66)
#define AW_PID_2071_EFRH_REG		(0x78)
#define AW_PID_2071_EFRM2_REG		(0x79)
#define AW_PID_2071_EFRM1_REG		(0x7A)

#define AW_PID_2071_MUTE_VOL (90 * 8)


/********************************************
 * Vcalb
 *******************************************/
#define AW_PID_2071_EF_VSN_GESLP_SIGN_MASK	(~0x0200)
#define AW_PID_2071_EF_VSN_GESLP_NEG		(~0xfc00)

#define AW_PID_2071_EF_ISN_GESLP_SIGN_MASK	(~0x0200)
#define AW_PID_2071_EF_ISN_GESLP_NEG		(~0xfc00)

#define AW_PID_2071_CABL_BASE_VALUE		(1000)
#define AW_PID_2071_ICABLK_FACTOR		(1)
#define AW_PID_2071_VCABLK_FACTOR		(1)

#define AW_PID_2071_VCAL_FACTOR			(1<<13)

#define AW_PID_2071_ICALK_SHIFT			(10)
#define AW_PID_2071_ICALKL_SHIFT		(10)
#define AW_PID_2071_HIGH_MASK			(0x03E0)
#define AW_PID_2071_LOW_MASK			(0x001F)

#define AW_PID_2071_MONITOR_VBAT_RANGE		(6025)
#define AW_PID_2071_MONITOR_INT_10BIT		(1023)
#define AW_PID_2071_MONITOR_TEMP_SIGN_MASK	(~(1<<9))
#define AW_PID_2071_MONITOR_TEMP_NEG_MASK	(0XFC00)

/* HMUTE bit 4 (SYSCTRL2 0x05) */
#define AW_PID_2071_HMUTE_START_BIT	(4)
#define AW_PID_2071_HMUTE_BITS_LEN	(1)
#define AW_PID_2071_HMUTE_MASK	\
	(~(((1<<AW_PID_2071_HMUTE_BITS_LEN)-1) << AW_PID_2071_HMUTE_START_BIT))

#define AW_PID_2071_HMUTE_DISABLE	(0)
#define AW_PID_2071_HMUTE_DISABLE_VALUE	\
	(AW_PID_2071_HMUTE_DISABLE << AW_PID_2071_HMUTE_START_BIT)

#define AW_PID_2071_HMUTE_ENABLE	(1)
#define AW_PID_2071_HMUTE_ENABLE_VALUE	\
	(AW_PID_2071_HMUTE_ENABLE << AW_PID_2071_HMUTE_START_BIT)

/* I2STXEN bit 0 (I2SCFG1 0x07) */
#define AW_PID_2071_I2STXEN_START_BIT	(0)
#define AW_PID_2071_I2STXEN_BITS_LEN	(1)
#define AW_PID_2071_I2STXEN_MASK	\
	(~(((1<<AW_PID_2071_I2STXEN_BITS_LEN)-1) << AW_PID_2071_I2STXEN_START_BIT))

#define AW_PID_2071_I2STXEN_DISABLE	(0)
#define AW_PID_2071_I2STXEN_DISABLE_VALUE	\
	(AW_PID_2071_I2STXEN_DISABLE << AW_PID_2071_I2STXEN_START_BIT)

#define AW_PID_2071_I2STXEN_ENABLE	(1)
#define AW_PID_2071_I2STXEN_ENABLE_VALUE	\
	(AW_PID_2071_I2STXEN_ENABLE << AW_PID_2071_I2STXEN_START_BIT)


/* EF_VSN_OFFSET bit 14:10 (EFRH 0x78) */
#define AW_PID_2071_EF_VSN_OFFSET_START_BIT	(10)
#define AW_PID_2071_EF_VSN_OFFSET_BITS_LEN	(5)
#define AW_PID_2071_EF_VSN_OFFSET_MASK	\
	(~(((1<<AW_PID_2071_EF_VSN_OFFSET_BITS_LEN)-1) << AW_PID_2071_EF_VSN_OFFSET_START_BIT))

/* EF_ISN_OFFSET bit 14:10 (EFRM1 0x7A) */
#define AW_PID_2071_EF_ISN_OFFSET_START_BIT	(10)
#define AW_PID_2071_EF_ISN_OFFSET_BITS_LEN	(5)
#define AW_PID_2071_EF_ISN_OFFSET_MASK	\
	(~(((1<<AW_PID_2071_EF_ISN_OFFSET_BITS_LEN)-1) << AW_PID_2071_EF_ISN_OFFSET_START_BIT))


/* EF_VSN_GESLP bit 9:0 (EFRH 0x78) */
#define AW_PID_2071_EF_VSN_GESLP_START_BIT	(0)
#define AW_PID_2071_EF_VSN_GESLP_BITS_LEN	(10)
#define AW_PID_2071_EF_VSN_GESLP_MASK	\
	(~(((1<<AW_PID_2071_EF_VSN_GESLP_BITS_LEN)-1) << AW_PID_2071_EF_VSN_GESLP_START_BIT))

/* CCO_MUX bit 14 (PLLCTRL1 0x66) */
#define AW_PID_2071_CCO_MUX_START_BIT	(14)
#define AW_PID_2071_CCO_MUX_BITS_LEN	(1)
#define AW_PID_2071_CCO_MUX_MASK		\
	(~(((1<<AW_PID_2071_CCO_MUX_BITS_LEN)-1) << AW_PID_2071_CCO_MUX_START_BIT))

#define AW_PID_2071_CCO_MUX_DIVIDED		(0)
#define AW_PID_2071_CCO_MUX_DIVIDED_VALUE	\
	(AW_PID_2071_CCO_MUX_DIVIDED << AW_PID_2071_CCO_MUX_START_BIT)

#define AW_PID_2071_CCO_MUX_BYPASS		(1)
#define AW_PID_2071_CCO_MUX_BYPASS_VALUE	\
	(AW_PID_2071_CCO_MUX_BYPASS << AW_PID_2071_CCO_MUX_START_BIT)

/* BST_IPEAK bit 3:0 (SYSCTRL2 0x05) */
#define AW_PID_2071_BST_IPEAK_START_BIT	(0)
#define AW_PID_2071_BST_IPEAK_BITS_LEN	(4)
#define AW_PID_2071_BST_IPEAK_MASK	\
	(~(((1<<AW_PID_2071_BST_IPEAK_BITS_LEN)-1) << AW_PID_2071_BST_IPEAK_START_BIT))

/* VOL bit 15:6 (SYSCTRL2 0x05) */
#define AW_PID_2071_VOL_START_BIT	(6)
#define AW_PID_2071_VOL_BITS_LEN	(10)
#define AW_PID_2071_VOL_MASK	\
	(~(((1<<AW_PID_2071_VOL_BITS_LEN)-1) << AW_PID_2071_VOL_START_BIT))

/********************************************
 * Dither
 *******************************************/
#define AW_PID_2071_DITHER_START_BIT	(13)
#define AW_PID_2071_DITHER_BITS_LEN	(1)
#define AW_PID_2071_DITHER_MASK	\
	(~(((1<<AW_PID_2071_DITHER_BITS_LEN)-1) << AW_PID_2071_DITHER_START_BIT))

#define AW_PID_2071_DITHER_DISABLE	(0)
#define AW_PID_2071_DITHER_DISABLE_VALUE	\
	(AW_PID_2071_DITHER_DISABLE << AW_PID_2071_DITHER_START_BIT)

#define AW_PID_2071_DITHER_ENABLE	(1)
#define AW_PID_2071_DITHER_ENABLE_VALUE	\
	(AW_PID_2071_DITHER_ENABLE << AW_PID_2071_DITHER_START_BIT)

#endif //__AW882XX_PID_2071_REG_H__

#ifndef __AW882XX_PID_2113_REG_H__
#define __AW882XX_PID_2113_REG_H__

#define  AW_PID_2113_REG_MAX (0x80)
#define AW_PID_2113_MONITOR_FILE	"aw882xx_pid_2113_monitor.bin"

#define AW_PID_2113_SYSCTRL_REG		(0x04)
#define AW_PID_2113_I2SCTRL3_REG	(0x08)
#define AW_PID_2113_DBGCTRL_REG		(0x14)
#define AW_PID_2113_SADCCTRL3_REG	(0x3C)

#define AW_PID_2113_VSNTM1_REG		(0x50)
#define AW_PID_2113_PLLCTRL1_REG	(0x54)

#define AW_PID_2113_BSTCTRL3_REG	(0x62)
#define AW_PID_2113_BSTCTRL5_REG	(0x64)
#define AW_PID_2113_BSTCTRL6_REG	(0x65)
#define AW_PID_2113_BSTCTRL7_REG	(0x66)
#define AW_PID_2113_BSTCTRL8_REG	(0x67)
#define AW_PID_2113_BSTCTRL9_REG	(0x68)

#define AW_PID_2113_EFRH4_REG		(0x78)
#define AW_PID_2113_EFRH3_REG		(0x79)

#define AW_PID_2113_EFRL4_REG		(0x7C)
#define AW_PID_2113_EFRL3_REG		(0x7D)

#define AW_PID_2113_INIT_CHECK_REG		(0x73)

#define AW_PID_2113_INIT_CHECK_VALUE		(0x1D40)

#define AW_PID_2113_MUTE_VOL (90 * 8)

/********************************************
 * frcset
 *******************************************/
#define AW_PID_2113_TEMH_MASK			(0x83ff)
#define AW_PID_2113_TEML_MASK			(0x83ff)
#define AW_PID_2113_DEFAULT_CFG			(0x0000)


/********************************************
 * Efdbmd
 *******************************************/
#define AW_PID_2113_EF_DBMD_MASK		(0xfff7)
#define AW_PID_2113_AND_VALUE			(0x0000)
#define AW_PID_2113_OR_VALUE			(0x0008)

/********************************************
 * Vcalb
 *******************************************/
#define AW_PID_2113_EF_VSN_GESLP_SIGN_MASK	(~0x0200)
#define AW_PID_2113_EF_VSN_GESLP_NEG		(~0xfc00)

#define AW_PID_2113_EF_ISN_GESLP_SIGN_MASK	(~0x0200)
#define AW_PID_2113_EF_ISN_GESLP_NEG		(~0xfc00)

#define AW_PID_2113_CABL_BASE_VALUE		(1000)
#define AW_PID_2113_ICABLK_FACTOR		(1)
#define AW_PID_2113_VCABLK_FACTOR		(1)

#define AW_PID_2113_VCAL_FACTOR			(1<<13)

#define AW_PID_2113_ICALK_SHIFT			(0)
#define AW_PID_2113_ICALKL_SHIFT		(0)
#define AW_PID_2113_VCALK_SHIFT			(0)
#define AW_PID_2113_VCALKL_SHIFT		(0)


/* ULS_HMUTE bit 14 (SYSCTRL 0x04) */
#define AW_PID_2113_ULS_HMUTE_START_BIT	(14)
#define AW_PID_2113_ULS_HMUTE_BITS_LEN	(1)
#define AW_PID_2113_ULS_HMUTE_MASK	\
	(~(((1<<AW_PID_2113_ULS_HMUTE_BITS_LEN)-1) << AW_PID_2113_ULS_HMUTE_START_BIT))

#define AW_PID_2113_ULS_HMUTE_DISABLE	(0)
#define AW_PID_2113_ULS_HMUTE_DISABLE_VALUE	\
	(AW_PID_2113_ULS_HMUTE_DISABLE << AW_PID_2113_ULS_HMUTE_START_BIT)

#define AW_PID_2113_ULS_HMUTE_ENABLE	(1)
#define AW_PID_2113_ULS_HMUTE_ENABLE_VALUE	\
	(AW_PID_2113_ULS_HMUTE_ENABLE << AW_PID_2113_ULS_HMUTE_START_BIT)

/* I2STXEN bit 6 (I2SCTRL3 0x08) */
#define AW_PID_2113_I2STXEN_START_BIT	(6)
#define AW_PID_2113_I2STXEN_BITS_LEN	(1)
#define AW_PID_2113_I2STXEN_MASK	\
	(~(((1<<AW_PID_2113_I2STXEN_BITS_LEN)-1) << AW_PID_2113_I2STXEN_START_BIT))

#define AW_PID_2113_I2STXEN_DISABLE	(0)
#define AW_PID_2113_I2STXEN_DISABLE_VALUE	\
	(AW_PID_2113_I2STXEN_DISABLE << AW_PID_2113_I2STXEN_START_BIT)

#define AW_PID_2113_I2STXEN_ENABLE	(1)
#define AW_PID_2113_I2STXEN_ENABLE_VALUE	\
	(AW_PID_2113_I2STXEN_ENABLE << AW_PID_2113_I2STXEN_START_BIT)


/* EF_ISN_GESLP_H bit 9:0 (EFRH4 0x78) */
#define AW_PID_2113_EF_ISN_GESLP_H_START_BIT	(0)
#define AW_PID_2113_EF_ISN_GESLP_H_BITS_LEN	(10)
#define AW_PID_2113_EF_ISN_GESLP_H_MASK	\
	(~(((1<<AW_PID_2113_EF_ISN_GESLP_H_BITS_LEN)-1) << AW_PID_2113_EF_ISN_GESLP_H_START_BIT))

/* EF_ISN_GESLP_L bit 9:0 (EFRL4 0x7C) */
#define AW_PID_2113_EF_ISN_GESLP_L_START_BIT	(0)
#define AW_PID_2113_EF_ISN_GESLP_L_BITS_LEN	(10)
#define AW_PID_2113_EF_ISN_GESLP_L_MASK	\
	(~(((1<<AW_PID_2113_EF_ISN_GESLP_L_BITS_LEN)-1) << AW_PID_2113_EF_ISN_GESLP_L_START_BIT))

/* EF_VSN_GESLP_H bit 9:0 (EFRH3 0x79) */
#define AW_PID_2113_EF_VSN_GESLP_H_START_BIT	(0)
#define AW_PID_2113_EF_VSN_GESLP_H_BITS_LEN	(10)
#define AW_PID_2113_EF_VSN_GESLP_H_MASK	\
	(~(((1<<AW_PID_2113_EF_VSN_GESLP_H_BITS_LEN)-1) << AW_PID_2113_EF_VSN_GESLP_H_START_BIT))

/* EF_VSN_GESLP_L bit 9:0 (EFRL3 0x7D) */
#define AW_PID_2113_EF_VSN_GESLP_L_START_BIT	(0)
#define AW_PID_2113_EF_VSN_GESLP_L_BITS_LEN	(10)
#define AW_PID_2113_EF_VSN_GESLP_L_MASK	\
	(~(((1<<AW_PID_2113_EF_VSN_GESLP_L_BITS_LEN)-1) << AW_PID_2113_EF_VSN_GESLP_L_START_BIT))

/* CCO_MUX bit 6 (PLLCTRL1 0x54) */
#define AW_PID_2113_CCO_MUX_START_BIT	(6)
#define AW_PID_2113_CCO_MUX_BITS_LEN	(1)
#define AW_PID_2113_CCO_MUX_MASK	\
	(~(((1<<AW_PID_2113_CCO_MUX_BITS_LEN)-1) << AW_PID_2113_CCO_MUX_START_BIT))

#define AW_PID_2113_CCO_MUX_DIVIDED	(0)
#define AW_PID_2113_CCO_MUX_DIVIDED_VALUE	\
		(AW_PID_2113_CCO_MUX_DIVIDED << AW_PID_2113_CCO_MUX_START_BIT)

#define AW_PID_2113_CCO_MUX_BYPASS	(1)
#define AW_PID_2113_CCO_MUX_BYPASS_VALUE	\
	(AW_PID_2113_CCO_MUX_BYPASS << AW_PID_2113_CCO_MUX_START_BIT)

/* BOP_EN bit 5 (SADCCTRL3 0x3C) */
#define AW_PID_2113_BOP_EN_START_BIT	(5)
#define AW_PID_2113_BOP_EN_BITS_LEN	(1)
#define AW_PID_2113_BOP_EN_MASK	\
	(~(((1<<AW_PID_2113_BOP_EN_BITS_LEN)-1) << AW_PID_2113_BOP_EN_START_BIT))

#define AW_PID_2113_BOP_EN_ENABLE	(1)
#define AW_PID_2113_BOP_EN_ENABLE_VALUE	\
	(AW_PID_2113_BOP_EN_ENABLE << AW_PID_2113_BOP_EN_START_BIT)

#define AW_PID_2113_BOP_EN_DISABLE	(0)
#define AW_PID_2113_BOP_EN_DISABLE_VALUE	\
	(AW_PID_2113_BOP_EN_DISABLE << AW_PID_2113_BOP_EN_START_BIT)

/********************************************
 * Dither
 *******************************************/
#define AW_PID_2113_DITHER_START_BIT	(14)
#define AW_PID_2113_DITHER_BITS_LEN	(1)
#define AW_PID_2113_DITHER_MASK	\
	(~(((1<<AW_PID_2113_DITHER_BITS_LEN)-1) << AW_PID_2113_DITHER_START_BIT))

#define AW_PID_2113_DITHER_DISABLE	(0)
#define AW_PID_2113_DITHER_DISABLE_VALUE	\
	(AW_PID_2113_DITHER_DISABLE << AW_PID_2113_DITHER_START_BIT)

#define AW_PID_2113_DITHER_ENABLE	(1)
#define AW_PID_2113_DITHER_ENABLE_VALUE	\
	(AW_PID_2113_DITHER_ENABLE << AW_PID_2113_DITHER_START_BIT)


/* FORCE_PWM bit 12 (BSTCTRL3 0x62) */
#define AW_PID_2113_FORCE_PWM_START_BIT	(12)
#define AW_PID_2113_FORCE_PWM_BITS_LEN	(1)
#define AW_PID_2113_FORCE_PWM_MASK	\
	(~(((1<<AW_PID_2113_FORCE_PWM_BITS_LEN)-1) << AW_PID_2113_FORCE_PWM_START_BIT))

#define AW_PID_2113_FORCE_PWM_FORCEMINUS_PWM	(1)
#define AW_PID_2113_FORCE_PWM_FORCEMINUS_PWM_VALUE	\
	(AW_PID_2113_FORCE_PWM_FORCEMINUS_PWM << AW_PID_2113_FORCE_PWM_START_BIT)

/* BST_OS_WIDTH bit 2:0 (BSTCTRL5 0x64) */
#define AW_PID_2113_BST_OS_WIDTH_START_BIT	(0)
#define AW_PID_2113_BST_OS_WIDTH_BITS_LEN	(3)
#define AW_PID_2113_BST_OS_WIDTH_MASK	\
	(~(((1<<AW_PID_2113_BST_OS_WIDTH_BITS_LEN)-1) << AW_PID_2113_BST_OS_WIDTH_START_BIT))


#define AW_PID_2113_BST_OS_WIDTH_50NS	(4)
#define AW_PID_2113_BST_OS_WIDTH_50NS_VALUE	\
	(AW_PID_2113_BST_OS_WIDTH_50NS << AW_PID_2113_BST_OS_WIDTH_START_BIT)

/* BST_LOOPR bit 1:0 (BSTCTRL6 0x65) */
#define AW_PID_2113_BST_LOOPR_START_BIT	(0)
#define AW_PID_2113_BST_LOOPR_BITS_LEN	(2)
#define AW_PID_2113_BST_LOOPR_MASK	\
	(~(((1<<AW_PID_2113_BST_LOOPR_BITS_LEN)-1) << AW_PID_2113_BST_LOOPR_START_BIT))

#define AW_PID_2113_BST_LOOPR_340K	(2)
#define AW_PID_2113_BST_LOOPR_340K_VALUE	\
	(AW_PID_2113_BST_LOOPR_340K << AW_PID_2113_BST_LOOPR_START_BIT)

/* RSQN_DLY bit 15:14 (BSTCTRL7 0x66) */
#define AW_PID_2113_RSQN_DLY_START_BIT	(14)
#define AW_PID_2113_RSQN_DLY_BITS_LEN	(2)
#define AW_PID_2113_RSQN_DLY_MASK	\
	(~(((1<<AW_PID_2113_RSQN_DLY_BITS_LEN)-1) << AW_PID_2113_RSQN_DLY_START_BIT))

#define AW_PID_2113_RSQN_DLY_35NS	(2)
#define AW_PID_2113_RSQN_DLY_35NS_VALUE	\
	(AW_PID_2113_RSQN_DLY_35NS << AW_PID_2113_RSQN_DLY_START_BIT)

/* BURST_SSMODE bit 3 (BSTCTRL8 0x67) */
#define AW_PID_2113_BURST_SSMODE_START_BIT	(3)
#define AW_PID_2113_BURST_SSMODE_BITS_LEN	(1)
#define AW_PID_2113_BURST_SSMODE_MASK	\
	(~(((1<<AW_PID_2113_BURST_SSMODE_BITS_LEN)-1) << AW_PID_2113_BURST_SSMODE_START_BIT))

#define AW_PID_2113_BURST_SSMODE_FAST	(0)
#define AW_PID_2113_BURST_SSMODE_FAST_VALUE	\
	(AW_PID_2113_BURST_SSMODE_FAST << AW_PID_2113_BURST_SSMODE_START_BIT)

/* BST_BURST bit 9:7 (BSTCTRL9 0x68) */
#define AW_PID_2113_BST_BURST_START_BIT	(7)
#define AW_PID_2113_BST_BURST_BITS_LEN	(3)
#define AW_PID_2113_BST_BURST_MASK	\
	(~(((1<<AW_PID_2113_BST_BURST_BITS_LEN)-1) << AW_PID_2113_BST_BURST_START_BIT))

#define AW_PID_2113_BST_BURST_30MA	(2)
#define AW_PID_2113_BST_BURST_30MA_VALUE	\
	(AW_PID_2113_BST_BURST_30MA << AW_PID_2113_BST_BURST_START_BIT)

#endif //__AW882XX_PID_2113_REG_H__



#ifndef __AW882XX_PID_2116_REG_H__
#define __AW882XX_PID_2116_REG_H__

#define AW_PID_2116_REG_MAX (0x72)

/* default value of SYSINTM (0x03) */
#define AW_PID_2116_SYSINTM_DEFAULT		(0x61FF)


#define AW_PID_2116_SYSCTRL_REG			(0x04)
#define AW_PID_2116_SYSCTRL2_REG		(0x05)
#define AW_PID_2116_I2SCTRL3_REG		(0x08)
#define AW_PID_2116_PWMCTRL4_REG		(0x16)
#define AW_PID_2116_PLLCTRL1_REG		(0x54)


/* EN_PA bit 1 (SYSCTRL 0x04) */
#define AW_PID_2116_EN_PA_START_BIT		(1)
#define AW_PID_2116_EN_PA_BITS_LEN		(1)
#define AW_PID_2116_EN_PA_MASK			\
		(~(((1<<AW_PID_2116_EN_PA_BITS_LEN)-1) << AW_PID_2116_EN_PA_START_BIT))

#define AW_PID_2116_EN_PA_WORKING	(1)
#define AW_PID_2116_EN_PA_WORKING_VALUE	\
		(AW_PID_2116_EN_PA_WORKING << AW_PID_2116_EN_PA_START_BIT)

#define AW_PID_2116_EN_PA_POWER_DOWN	(0)
#define AW_PID_2116_EN_PA_POWER_DOWN_VALUE	\
		(AW_PID_2116_EN_PA_POWER_DOWN << AW_PID_2116_EN_PA_START_BIT)

/* ULS_HMUTE bit 14 (SYSCTRL 0x04) */
#define AW_PID_2116_ULS_HMUTE_START_BIT	(14)
#define AW_PID_2116_ULS_HMUTE_BITS_LEN	(1)
#define AW_PID_2116_ULS_HMUTE_MASK		\
		(~(((1<<AW_PID_2116_ULS_HMUTE_BITS_LEN)-1) << AW_PID_2116_ULS_HMUTE_START_BIT))

#define AW_PID_2116_ULS_HMUTE_DISABLE	(0)
#define AW_PID_2116_ULS_HMUTE_DISABLE_VALUE	\
		(AW_PID_2116_ULS_HMUTE_DISABLE << AW_PID_2116_ULS_HMUTE_START_BIT)

#define AW_PID_2116_ULS_HMUTE_ENABLE	(1)
#define AW_PID_2116_ULS_HMUTE_ENABLE_VALUE	\
		(AW_PID_2116_ULS_HMUTE_ENABLE << AW_PID_2116_ULS_HMUTE_START_BIT)

/* I2STXEN bit 6 (I2SCTRL3 0x08) */
#define AW_PID_2116_I2STXEN_START_BIT	(6)
#define AW_PID_2116_I2STXEN_BITS_LEN	(1)
#define AW_PID_2116_I2STXEN_MASK		\
		(~(((1<<AW_PID_2116_I2STXEN_BITS_LEN)-1) << AW_PID_2116_I2STXEN_START_BIT))

#define AW_PID_2116_I2STXEN_DISABLE		(0)
#define AW_PID_2116_I2STXEN_DISABLE_VALUE	\
		(AW_PID_2116_I2STXEN_DISABLE << AW_PID_2116_I2STXEN_START_BIT)

#define AW_PID_2116_I2STXEN_ENABLE		(1)
#define AW_PID_2116_I2STXEN_ENABLE_VALUE	\
		(AW_PID_2116_I2STXEN_ENABLE << AW_PID_2116_I2STXEN_START_BIT)

/* CCO_MUX bit 13 (PLLCTRL1 0x54) */
#define AW_PID_2116_CCO_MUX_START_BIT	(13)
#define AW_PID_2116_CCO_MUX_BITS_LEN	(1)
#define AW_PID_2116_CCO_MUX_MASK		\
		(~(((1<<AW_PID_2116_CCO_MUX_BITS_LEN)-1) << AW_PID_2116_CCO_MUX_START_BIT))

#define AW_PID_2116_CCO_MUX_DIVIDED		(0)
#define AW_PID_2116_CCO_MUX_DIVIDED_VALUE	\
		(AW_PID_2116_CCO_MUX_DIVIDED << AW_PID_2116_CCO_MUX_START_BIT)

#define AW_PID_2116_CCO_MUX_BYPASS		(1)
#define AW_PID_2116_CCO_MUX_BYPASS_VALUE	\
		(AW_PID_2116_CCO_MUX_BYPASS << AW_PID_2116_CCO_MUX_START_BIT)

/* BOP_EN bit 14 (SYSCTRL2 0x05) */
#define AW_PID_2116_BOP_EN_START_BIT	(14)
#define AW_PID_2116_BOP_EN_BITS_LEN		(1)
#define AW_PID_2116_BOP_EN_MASK			\
		(~(((1<<AW_PID_2116_BOP_EN_BITS_LEN)-1) << AW_PID_2116_BOP_EN_START_BIT))

#define AW_PID_2116_BOP_EN_DISABLE		(0)
#define AW_PID_2116_BOP_EN_DISABLE_VALUE	\
		(AW_PID_2116_BOP_EN_DISABLE << AW_PID_2116_BOP_EN_START_BIT)

#define AW_PID_2116_BOP_EN_ENABLE		(1)
#define AW_PID_2116_BOP_EN_ENABLE_VALUE	\
		(AW_PID_2116_BOP_EN_ENABLE << AW_PID_2116_BOP_EN_START_BIT)

/********************************************
 * NOISE_GATE_EN bit 13 (PWMCTRL4 0x16) *
 *******************************************/
#define AW_PID_2116_NOISE_GATE_EN_START_BIT	(13)
#define AW_PID_2116_NOISE_GATE_EN_BITS_LEN	(1)
#define AW_PID_2116_NOISE_GATE_EN_MASK	\
			(~(((1<<AW_PID_2116_NOISE_GATE_EN_BITS_LEN)-1) << AW_PID_2116_NOISE_GATE_EN_START_BIT))

#define AW_PID_2116_NOISE_GATE_EN_DISABLE	(0)
#define AW_PID_2116_NOISE_GATE_EN_DISABLE_VALUE	\
		(AW_PID_2116_NOISE_GATE_EN_DISABLE << AW_PID_2116_NOISE_GATE_EN_START_BIT)

#define AW_PID_2116_NOISE_GATE_EN_ENABLE	(1)
#define AW_PID_2116_NOISE_GATE_EN_ENABLE_VALUE	\
		(AW_PID_2116_NOISE_GATE_EN_ENABLE << AW_PID_2116_NOISE_GATE_EN_START_BIT)

#define AW_PID_2116_SYSST_CHECK_MASK \
	(~(AW_UVLS_UVLO_VALUE | \
	AW_SWS_SWITCHING_VALUE | \
	AW_NOCLKS_NO_CLOCK_VALUE | \
	AW_CLKS_STABLE_VALUE | \
	AW_OCDS_OC_VALUE | \
	AW_OTHS_OT_VALUE | \
	AW_PLLS_LOCKED_VALUE))

#define AW_PID_2116_NO_SWS_SYSST_CHECK \
		(AW_CLKS_STABLE_VALUE | \
		AW_PLLS_LOCKED_VALUE)

#define AW_PID_2116_SWS_SYSST_CHECK \
	(AW_SWS_SWITCHING_VALUE | \
	AW_CLKS_STABLE_VALUE | \
	AW_PLLS_LOCKED_VALUE)

#endif //__AW882XX_PID_2116_REG_H__


#ifndef __AW882XX_PID_2308_REG_H__
#define __AW882XX_PID_2308_REG_H__

#define AW_PID_2308_REG_MAX (0x80)

#define AW_PID_2308_MONITOR_FILE	"aw882xx_pid_2308_monitor.bin"

#define AW_PID_2308_SYSCTRL2_REG		(0x05)
#define AW_PID_2308_TESTOUT_REG			(0x2E)
#define AW_PID_2308_TESTIN_REG			(0x38)
#define AW_PID_2308_NGCTRL3_REG			(0x5E)
#define AW_PID_2308_EFRH2_REG			(0x7A)

/* PSM_EN bit 14 (SYSCTRL2 0x05) */
#define AW_PID_2308_PSM_EN_START_BIT	(14)
#define AW_PID_2308_PSM_EN_BITS_LEN		(1)
#define AW_PID_2308_PSM_EN_MASK			\
		(~(((1<<AW_PID_2308_PSM_EN_BITS_LEN)-1) << AW_PID_2308_PSM_EN_START_BIT))

#define AW_PID_2308_PSM_EN_ENABLE		(1)
#define AW_PID_2308_PSM_EN_ENABLE_VALUE	\
		(AW_PID_2308_PSM_EN_ENABLE << AW_PID_2308_PSM_EN_START_BIT)

#define AW_PID_2308_PSM_EN_DISABLE		(0)
#define AW_PID_2308_PSM_EN_DISABLE_VALUE	\
		(AW_PID_2308_PSM_EN_DISABLE << AW_PID_2308_PSM_EN_START_BIT)

/* EN_MPD bit 15 (SYSCTRL2 0x05) */
#define AW_PID_2308_EN_MPD_START_BIT	(15)
#define AW_PID_2308_EN_MPD_BITS_LEN		(1)
#define AW_PID_2308_EN_MPD_MASK			\
		(~(((1<<AW_PID_2308_EN_MPD_BITS_LEN)-1) << AW_PID_2308_EN_MPD_START_BIT))

#define AW_PID_2308_EN_MPD_DISABLE		(0)
#define AW_PID_2308_EN_MPD_DISABLE_VALUE	\
		(AW_PID_2308_EN_MPD_DISABLE << AW_PID_2308_EN_MPD_START_BIT)

#define AW_PID_2308_EN_MPD_ENABLE		(1)
#define AW_PID_2308_EN_MPD_ENABLE_VALUE	\
		(AW_PID_2308_EN_MPD_ENABLE << AW_PID_2308_EN_MPD_START_BIT)

#define AW_PID_2308_EN_MPD_DEFAULT		(1)
#define AW_PID_2308_EN_MPD_DEFAULT_VALUE	\
		(AW_PID_2308_EN_MPD_DEFAULT << AW_PID_2308_EN_MPD_START_BIT)

/* DSMZTH bit 7:4 (NGCTRL3 0x5E) */
#define AW_PID_2308_DSMZTH_START_BIT	(4)
#define AW_PID_2308_DSMZTH_BITS_LEN		(4)
#define AW_PID_2308_DSMZTH_MASK			\
		(~(((1<<AW_PID_2308_DSMZTH_BITS_LEN)-1) << AW_PID_2308_DSMZTH_START_BIT))

#define AW_PID_2308_DSMZTH_NO_RESET		(0)
#define AW_PID_2308_DSMZTH_NO_RESET_VALUE	\
		(AW_PID_2308_DSMZTH_NO_RESET << AW_PID_2308_DSMZTH_START_BIT)

#define AW_PID_2308_DSMZTH_21P33MS		(5)
#define AW_PID_2308_DSMZTH_21P33MS_VALUE	\
		(AW_PID_2308_DSMZTH_21P33MS << AW_PID_2308_DSMZTH_START_BIT)

#endif //__AW882XX_PID_2308_REG_H__


#ifndef __AW882XX_PID_2407_REG_H__
#define __AW882XX_PID_2407_REG_H__

#define AW_PID_2407_I2SCTRL2_REG	(0x07)

#define AW_PID_2407_LOCKH_REG (0x76)
#define AW_PID_2407_LOCKL_REG (0x78)


#define AW_PID_2407_REG_MAX			(0x7E)
#define AW_PID_2407_MONITOR_FILE	"aw882xx_pid_2407_monitor.bin"


#define AW_PID_2407_SYSST_CHECK_MASK \
	(~(AW_UVLS_UVLO_VALUE | \
	AW_SWS_SWITCHING_VALUE | \
	AW_NOCLKS_NO_CLOCK_VALUE | \
	AW_CLKS_STABLE_VALUE | \
	AW_OCDS_OC_VALUE | \
	AW_OTHS_OT_VALUE | \
	AW_PLLS_LOCKED_VALUE))

#define AW_PID_2407_NO_SWS_SYSST_CHECK \
		(AW_CLKS_STABLE_VALUE | \
		AW_PLLS_LOCKED_VALUE)

#define AW_PID_2407_SWS_SYSST_CHECK \
	(AW_SWS_SWITCHING_VALUE | \
	AW_CLKS_STABLE_VALUE | \
	AW_PLLS_LOCKED_VALUE)

#define AW_PID_2407_IIS_CHECK \
	(AW_CLKS_STABLE_VALUE | \
	AW_PLLS_LOCKED_VALUE)


/* SYSCTRL (0x04) detail */
/* I2C_WEN bit 15:14 (SYSCTRL 0x04) */
#define AW_PID_2407_I2C_WEN_START_BIT	(14)
#define AW_PID_2407_I2C_WEN_BITS_LEN	(2)
#define AW_PID_2407_I2C_WEN_MASK	\
	(~(((1<<AW_PID_2407_I2C_WEN_BITS_LEN)-1) << AW_PID_2407_I2C_WEN_START_BIT))

#define AW_PID_2407_I2C_WEN_ENABLE	(2)
#define AW_PID_2407_I2C_WEN_ENABLE_VALUE	\
	(AW_PID_2407_I2C_WEN_ENABLE << AW_PID_2407_I2C_WEN_START_BIT)

#define AW_PID_2407_I2C_WEN_DISABLE	(0)
#define AW_PID_2407_I2C_WEN_DISABLE_VALUE	\
	(AW_PID_2407_I2C_WEN_DISABLE << AW_PID_2407_I2C_WEN_START_BIT)

/* I2STXEN bit 7 (SYSCTRL 0x04) */
#define AW_PID_2407_I2STXEN_START_BIT	(7)
#define AW_PID_2407_I2STXEN_BITS_LEN	(1)
#define AW_PID_2407_I2STXEN_MASK	\
	(~(((1<<AW_PID_2407_I2STXEN_BITS_LEN)-1) << AW_PID_2407_I2STXEN_START_BIT))

#define AW_PID_2407_I2STXEN_DISABLE	(0)
#define AW_PID_2407_I2STXEN_DISABLE_VALUE	\
	(AW_PID_2407_I2STXEN_DISABLE << AW_PID_2407_I2STXEN_START_BIT)

#define AW_PID_2407_I2STXEN_ENABLE	(1)
#define AW_PID_2407_I2STXEN_ENABLE_VALUE	\
	(AW_PID_2407_I2STXEN_ENABLE << AW_PID_2407_I2STXEN_START_BIT)


#endif //__AW882XX_PID_2407_REG_H__


#ifndef __AW882XX_PID_2329_REG_H__
#define __AW882XX_PID_2329_REG_H__
#define AW_PID_2329_REG_MAX			(0x80)
#define AW_PID_2329_MONITOR_FILE	"aw882xx_pid_2329_monitor.bin"

#define AW_PID_2329_BSTCTRL1_REG		(0x60)
#define AW_PID_2329_LPC_REG				(0x59)

/* I2C_WEN bit 11:10 (SYSCTRL 0x04) */
#define AW_PID_2329_I2C_WEN_START_BIT	(10)
#define AW_PID_2329_I2C_WEN_BITS_LEN	(2)
#define AW_PID_2329_I2C_WEN_MASK	\
	(~(((1<<AW_PID_2329_I2C_WEN_BITS_LEN)-1) << AW_PID_2329_I2C_WEN_START_BIT))

#define AW_PID_2329_I2C_WEN_ENABLE	(2)
#define AW_PID_2329_I2C_WEN_ENABLE_VALUE	\
	(AW_PID_2329_I2C_WEN_ENABLE << AW_PID_2329_I2C_WEN_START_BIT)

#define AW_PID_2329_I2C_WEN_DISABLE	(0)
#define AW_PID_2329_I2C_WEN_DISABLE_VALUE	\
	(AW_PID_2329_I2C_WEN_DISABLE << AW_PID_2329_I2C_WEN_START_BIT)

/* BST_IPEAK bit 15:10 (BSTCTRL2 0x60) */
#define AW_PID_2329_BST_IPEAK_START_BIT	(10)
#define AW_PID_2329_BST_IPEAK_BITS_LEN	(6)
#define AW_PID_2329_BST_IPEAK_MASK		\
	(~(((1<<AW_PID_2329_BST_IPEAK_BITS_LEN)-1) << AW_PID_2329_BST_IPEAK_START_BIT))

/* LPC_DET_EN bit 5 (LPCCTRL1 0x59) */
#define AW_PID_2329_LPC_DETEN_START_BIT		(5)
#define AW_PID_2329_LPC_DETEN_BITS_LEN		(1)
#define AW_PID_2329_LPC_DETEN_MASK	\
	(~(((1<<AW_PID_2329_LPC_DETEN_BITS_LEN)-1) << AW_PID_2329_LPC_DETEN_START_BIT))

#define AW_PID_2329_LPC_DETEN_DISABLE		(0)
#define AW_PID_2329_LPC_DETEN_DISABLE_VALUE	\
	(AW_PID_2329_LPC_DETEN_DISABLE << AW_PID_2329_LPC_DETEN_START_BIT)

#define AW_PID_2329_LPC_DETEN_ENABLE		(1)
#define AW_PID_2329_LPC_DETEN_ENABLE_VALUE	\
	(AW_PID_2329_LPC_DETEN_ENABLE << AW_PID_2329_LPC_DETEN_START_BIT)
#endif //__AW882XX_PID_2329_REG_H__


#ifndef __AW882XX_PID_2330_REG_H__
#define __AW882XX_PID_2330_REG_H__
#define AW_PID_2330_REG_MAX			(0x7B)
#define AW_PID_2330_MONITOR_FILE	"aw882xx_pid_2330_monitor.bin"

#define AW_PID_2330_BSTCTRL3_REG	(0x62)
#define AW_PID_2330_NOISE_GATE_REG	(0x5B)
#define AW_PID_2330_EF_LOCK_H_REG	(0x77)
#define AW_PID_2330_EF_LOCK_L_REG	(0x7A)

/* I2C_WEN bit 11:10 (SYSCTRL 0x04) */
#define AW_PID_2330_I2C_WEN_START_BIT	(10)
#define AW_PID_2330_I2C_WEN_BITS_LEN	(2)
#define AW_PID_2330_I2C_WEN_MASK	\
	(~(((1<<AW_PID_2330_I2C_WEN_BITS_LEN)-1) << AW_PID_2330_I2C_WEN_START_BIT))

#define AW_PID_2330_I2C_WEN_ENABLE	(2)
#define AW_PID_2330_I2C_WEN_ENABLE_VALUE	\
	(AW_PID_2330_I2C_WEN_ENABLE << AW_PID_2330_I2C_WEN_START_BIT)

#define AW_PID_2330_I2C_WEN_DISABLE	(0)
#define AW_PID_2330_I2C_WEN_DISABLE_VALUE	\
	(AW_PID_2330_I2C_WEN_DISABLE << AW_PID_2330_I2C_WEN_START_BIT)

/* BST_IPEAK bit 15:10 (BSTCTRL3 0x62) */
#define AW_PID_2330_BST_IPEAK_START_BIT	(10)
#define AW_PID_2330_BST_IPEAK_BITS_LEN	(6)
#define AW_PID_2330_BST_IPEAK_MASK		\
	(~(((1<<AW_PID_2330_BST_IPEAK_BITS_LEN)-1) << AW_PID_2330_BST_IPEAK_START_BIT))

#endif //__AW882XX_PID_2330_REG_H__

#ifndef __AW882XX_PID_2418_REG_H__
#define __AW882XX_PID_2418_REG_H__

#define AW_PID_2418_REG_MAX (0x80)

#define AW_PID_2418_MONITOR_FILE	"aw882xx_pid_2418_monitor.bin"

#define AW_PID_2418_SYSCTRL2_REG		(0x05)
#define AW_PID_2418_I2SCTRL3_REG		(0x08)
#define AW_PID_2418_VBAT_REG			(0x20)
#define AW_PID_2418_TEMP_REG			(0x21)
#define AW_PID_2418_PVDD_REG			(0x22)
#define AW_PID_2418_DBGCTRL_REG			(0x25)
#define AW_PID_2418_TESTOUT_REG			(0x2D)
#define AW_PID_2418_TESTIN_REG			(0x30)
#define AW_PID_2418_BSTCTRL14_REG		(0x42)
#define AW_PID_2418_NGCTRL3_REG			(0x52)
#define AW_PID_2418_EFCTRL2_REG			(0x75)
#define AW_PID_2418_EFRH2_REG			(0x76)
#define AW_PID_2418_LPC_REG				(0x3A)

/*monitor reg*/
#define AW_PID_2418_MONITOR_VBAT_RANGE		(6065)
#define AW_PID_2418_MONITOR_INT_10BIT		(1023)  // 1 << 10 -1
#define AW_PID_2418_MONITOR_TEMP_SIGN_MASK	(~(1<<9))
#define AW_PID_2418_MONITOR_TEMP_NEG_MASK	(0xFC00)

/* PSM_EN bit 14 (SYSCTRL2 0x05) */
#define AW_PID_2418_PSM_EN_START_BIT	(14)
#define AW_PID_2418_PSM_EN_BITS_LEN		(1)
#define AW_PID_2418_PSM_EN_MASK			\
		(~(((1<<AW_PID_2418_PSM_EN_BITS_LEN)-1) << AW_PID_2418_PSM_EN_START_BIT))

#define AW_PID_2418_PSM_EN_ENABLE		(1)
#define AW_PID_2418_PSM_EN_ENABLE_VALUE	\
		(AW_PID_2418_PSM_EN_ENABLE << AW_PID_2418_PSM_EN_START_BIT)

#define AW_PID_2418_PSM_EN_DISABLE		(0)
#define AW_PID_2418_PSM_EN_DISABLE_VALUE	\
		(AW_PID_2418_PSM_EN_DISABLE << AW_PID_2418_PSM_EN_START_BIT)

/* EN_MPD bit 15 (SYSCTRL2 0x05) */
#define AW_PID_2418_EN_MPD_START_BIT	(15)
#define AW_PID_2418_EN_MPD_BITS_LEN		(1)
#define AW_PID_2418_EN_MPD_MASK			\
		(~(((1<<AW_PID_2418_EN_MPD_BITS_LEN)-1) << AW_PID_2418_EN_MPD_START_BIT))

#define AW_PID_2418_EN_MPD_DISABLE		(0)
#define AW_PID_2418_EN_MPD_DISABLE_VALUE	\
		(AW_PID_2418_EN_MPD_DISABLE << AW_PID_2418_EN_MPD_START_BIT)

#define AW_PID_2418_EN_MPD_ENABLE		(1)
#define AW_PID_2418_EN_MPD_ENABLE_VALUE	\
		(AW_PID_2418_EN_MPD_ENABLE << AW_PID_2418_EN_MPD_START_BIT)

#define AW_PID_2418_EN_MPD_DEFAULT		(1)
#define AW_PID_2418_EN_MPD_DEFAULT_VALUE	\
		(AW_PID_2418_EN_MPD_DEFAULT << AW_PID_2418_EN_MPD_START_BIT)

/* I2STXEN bit 5 (I2SCTRL3 0x08) */
#define AW_PID_2418_I2STXEN_START_BIT	(5)
#define AW_PID_2418_I2STXEN_BITS_LEN	(1)
#define AW_PID_2418_I2STXEN_MASK	\
	(~(((1<<AW_PID_2418_I2STXEN_BITS_LEN)-1) << AW_PID_2418_I2STXEN_START_BIT))

#define AW_PID_2418_I2STXEN_DISABLE	(0)
#define AW_PID_2418_I2STXEN_DISABLE_VALUE	\
	(AW_PID_2418_I2STXEN_DISABLE << AW_PID_2418_I2STXEN_START_BIT)

#define AW_PID_2418_I2STXEN_ENABLE	(1)
#define AW_PID_2418_I2STXEN_ENABLE_VALUE	\
	(AW_PID_2418_I2STXEN_ENABLE << AW_PID_2418_I2STXEN_START_BIT)

/* DITHER_EN bit 15 (DBGCTRL 0x25) */
#define AW_PID_2418_DITHER_EN_START_BIT	(15)
#define AW_PID_2418_DITHER_EN_BITS_LEN	(1)
#define AW_PID_2418_DITHER_EN_MASK	\
	(~(((1<<AW_PID_2418_DITHER_EN_BITS_LEN)-1) << AW_PID_2418_DITHER_EN_START_BIT))

#define AW_PID_2418_DITHER_EN_DISABLE	(0)
#define AW_PID_2418_DITHER_EN_DISABLE_VALUE	\
	(AW_PID_2418_DITHER_EN_DISABLE << AW_PID_2418_DITHER_EN_START_BIT)

#define AW_PID_2418_DITHER_EN_ENABLE	(1)
#define AW_PID_2418_DITHER_EN_ENABLE_VALUE	\
	(AW_PID_2418_DITHER_EN_ENABLE << AW_PID_2418_DITHER_EN_START_BIT)

/* CCO_MUX bit 13 (DBGCTRL 0x25) */
#define AW_PID_2418_CCO_MUX_START_BIT	(13)
#define AW_PID_2418_CCO_MUX_BITS_LEN	(1)
#define AW_PID_2418_CCO_MUX_MASK	\
	(~(((1<<AW_PID_2418_CCO_MUX_BITS_LEN)-1) << AW_PID_2418_CCO_MUX_START_BIT))

#define AW_PID_2418_CCO_MUX_DIVIDED	(0)
#define AW_PID_2418_CCO_MUX_DIVIDED_VALUE	\
	(AW_PID_2418_CCO_MUX_DIVIDED << AW_PID_2418_CCO_MUX_START_BIT)

#define AW_PID_2418_CCO_MUX_BYPASS	(1)
#define AW_PID_2418_CCO_MUX_BYPASS_VALUE	\
	(AW_PID_2418_CCO_MUX_BYPASS << AW_PID_2418_CCO_MUX_START_BIT)

/* BSTCTRL14 (0x42) detail */
/* BST_IPEAK bit 15:11 (BSTCTRL14 0x42) */
#define AW_PID_2418_BST_IPEAK_START_BIT	(11)
#define AW_PID_2418_BST_IPEAK_BITS_LEN	(5)
#define AW_PID_2418_BST_IPEAK_MASK	\
	(~(((1<<AW_PID_2418_BST_IPEAK_BITS_LEN)-1) << AW_PID_2418_BST_IPEAK_START_BIT))

/* DSMZTH bit 7:4 (NGCTRL3 0x52) */
#define AW_PID_2418_DSMZTH_START_BIT	(4)
#define AW_PID_2418_DSMZTH_BITS_LEN		(4)
#define AW_PID_2418_DSMZTH_MASK			\
		(~(((1<<AW_PID_2418_DSMZTH_BITS_LEN)-1) << AW_PID_2418_DSMZTH_START_BIT))

#define AW_PID_2418_DSMZTH_NO_RESET		(0)
#define AW_PID_2418_DSMZTH_NO_RESET_VALUE	\
		(AW_PID_2418_DSMZTH_NO_RESET << AW_PID_2418_DSMZTH_START_BIT)

#define AW_PID_2418_DSMZTH_21P33MS		(5)
#define AW_PID_2418_DSMZTH_21P33MS_VALUE	\
		(AW_PID_2418_DSMZTH_21P33MS << AW_PID_2418_DSMZTH_START_BIT)

/* NOISE_GATE_EN bit 0 (NGCTRL3 0x52) */
#define AW_PID_2418_NOISE_GATE_EN_START_BIT	(0)
#define AW_PID_2418_NOISE_GATE_EN_BITS_LEN	(1)
#define AW_PID_2418_NOISE_GATE_EN_MASK	\
	(~(((1<<AW_PID_2418_NOISE_GATE_EN_BITS_LEN)-1) << AW_PID_2418_NOISE_GATE_EN_START_BIT))

#define AW_PID_2418_NOISE_GATE_EN_DISABLE	(0)
#define AW_PID_2418_NOISE_GATE_EN_DISABLE_VALUE	\
	(AW_PID_2418_NOISE_GATE_EN_DISABLE << AW_PID_2418_NOISE_GATE_EN_START_BIT)

#define AW_PID_2418_NOISE_GATE_EN_ENABLE	(1)
#define AW_PID_2418_NOISE_GATE_EN_ENABLE_VALUE	\
	(AW_PID_2418_NOISE_GATE_EN_ENABLE << AW_PID_2418_NOISE_GATE_EN_START_BIT)

/* EF_LOCK_TRIM bit 15 (EFRH2 0x76) */
#define AW_PID_2418_EF_LOCK_TRIM_START_BIT	(15)
#define AW_PID_2418_EF_LOCK_TRIM_BITS_LEN	(1)
#define AW_PID_2418_EF_LOCK_TRIM_MASK	\
	(~(((1<<AW_PID_2418_EF_LOCK_TRIM_BITS_LEN)-1) << AW_PID_2418_EF_LOCK_TRIM_START_BIT))

#define AW_PID_2418_EF_LOCK_TRIM_UNLOCKED	(0)
#define AW_PID_2418_EF_LOCK_TRIM_UNLOCKED_VALUE	\
	(AW_PID_2418_EF_LOCK_TRIM_UNLOCKED << AW_PID_2418_EF_LOCK_TRIM_START_BIT)

#define AW_PID_2418_EF_LOCK_TRIM_LOCKED	(1)
#define AW_PID_2418_EF_LOCK_TRIM_LOCKED_VALUE	\
	(AW_PID_2418_EF_LOCK_TRIM_LOCKED << AW_PID_2418_EF_LOCK_TRIM_START_BIT)

/* LPC_DET_EN bit 8 (POT_LPC1 0x3A) */
#define AW_PID_2418_LPC_DETEN_START_BIT		(8)
#define AW_PID_2418_LPC_DETEN_BITS_LEN		(1)
#define AW_PID_2418_LPC_DETEN_MASK	\
	(~(((1<<AW_PID_2418_LPC_DETEN_BITS_LEN)-1) << AW_PID_2418_LPC_DETEN_START_BIT))

#define AW_PID_2418_LPC_DETEN_DISABLE		(0)
#define AW_PID_2418_LPC_DETEN_DISABLE_VALUE	\
	(AW_PID_2418_LPC_DETEN_DISABLE << AW_PID_2418_LPC_DETEN_START_BIT)

#define AW_PID_2418_LPC_DETEN_ENABLE		(1)
#define AW_PID_2418_LPC_DETEN_ENABLE_VALUE	\
	(AW_PID_2418_LPC_DETEN_ENABLE << AW_PID_2418_LPC_DETEN_START_BIT)

#endif //__AW882XX_PID_2418_REG_H__

#endif /*init end*/

